#include <stdio.h>
#include <malloc.h>
#include <unistd.h>

/*
** Datatypes
*/

typedef enum bool
{
   False=0,
   True=1
}
bool;

typedef struct ColumnDesc
{
   char *Name;
   int Start;
   int Stop;
   char Type;

   struct ColumnDesc *Next;
}
ColumnDesc;

typedef struct CONFIG
{
   char *ProgramName;
   char *TableId;
   char *OutputBasename;
   char *DataFile;

   ColumnDesc *FirstColumn;
   ColumnDesc *LastColumn;
}
CONFIG;

void DoError( CONFIG *Config, char *File, int Line, char *Msg )
{
   fprintf( stderr, "%s - %s %d : %s\n",
            (Config->ProgramName==NULL)?"":Config->ProgramName,
	    File,
	    Line,
	    Msg );

}

void DoFatalError( CONFIG *Config, char *File, int Line, char *Msg )
{
   DoError( Config, File, Line, Msg );
   exit(1);
}

#define FatalError( msg ) DoFatalError( Config, __FILE__, __LINE__, (msg))

#define Error( msg ) DoError( Config, __FILE__, __LINE__, (msg))


char *DupString( CONFIG *Config, char *String )
{
   char *new;

   new = strdup( String );

   if ( new == NULL )
   {
      FatalError( "Memory allocation failed" );
   }

   return new;
}

void InitConfig( CONFIG *Config, char *ProgramName )
{
   memset( Config, 0, sizeof( CONFIG ));

   Config->ProgramName = DupString(Config, ProgramName);
}

void DisplayShortUsage( CONFIG *Config, FILE *Out )
{
   fprintf( Out, "TOBEDONE...\n" );
}

void DisplayUsage( CONFIG *Config, FILE *Out )
{
   fprintf( Out, "TOBEDONE...\n" );
}

void DisplayVersion( CONFIG *Config, FILE *Out )
{
   fprintf( Out, "TOBEDONE...\n" );
}


void ParseOpt( CONFIG *Config, int Argc, char **Argv )
{
   int c;
   int err;
   bool help;
   bool vers;
   extern char *optarg;
   extern int optind, opterr, optopt;

   err  = 0;         /* Error count      */
   help = False;     /* Help required    */
   vers = False;     /* Version required */

   while( (c=getopt( Argc, Argv, "hvo:t:d:" )) > 0 )
   {
      switch( c )
      {
         case 't':
            Config->TableId = DupString( Config, optarg );
	    break;

         case 'o':
            Config->OutputBasename = DupString( Config, optarg );
	    break;

         case 'd':
            Config->DataFile = DupString( Config, optarg );
	    break;

	 case 'v' :
	    vers = True;
	    ;;

	 case 'h' :
	    help = True;
	    break;

	 default:
            fprintf( stderr, "Invalid option %c\n", c );
	    ++err;
	    break;
      }
   }

   /*
   ** -t and -o options are mandatory
   */
   if ( Config->TableId == NULL || Config->OutputBasename == NULL )
   {
      Error( "TableId and OutputBasename are mandatory" );
      ++err;
   }

   /*
   ** Unknown or invalid option encountered
   */
   if ( err )
   {
      DisplayShortUsage( Config, stderr );
      exit( 1 );
   }

   /*
   ** Help required
   */
   if ( help )
   {
      DisplayUsage( Config, stdout );
      exit( 0 );
   }

   /*
   ** Version required
   */
   if ( vers )
   {
      DisplayVersion( Config, stdout );
      exit ( 0 );
   }
}

#define BUFFER 1024

char *GetField( CONFIG *Config, FILE *From )
{
   static char FieldValue[BUFFER+1];
   int i, c;
   static int eol = False;

   i = 0;

   if( eol ) return NULL;

   while ( (c=fgetc(From)) != EOF && c != ',' && c != '\n' && c >= '0')
   {
      FieldValue[i++] = c;

      if ( i > BUFFER )
      {
         FatalError( "Invalid format description line..." );
      }
   }

   FieldValue[i] = '\0';

   if ( c != ',' && (c < '0' || c == EOF) ) eol = True;

   if ( i > 0 ) return FieldValue;

   return NULL;
}

void ParseFormat( CONFIG *Config, FILE *From )
{
   char *field;
   ColumnDesc *new;

   if( fgetc(From) != '#' )
   {
      FatalError ( "Data format description is missing..." );
   }

   while( (field=GetField( Config, From )) != NULL )
   {
      new = (ColumnDesc *)malloc(sizeof(ColumnDesc));

      if( new == NULL )
      {
         FatalError( "Memory allocation problem..." );
      }

      new->Name = DupString( Config, field );

      field=GetField( Config, From );
      if ( field == NULL )
      {
         FatalError( "Invalid format description line..." );
      }
      new->Start = atoi( field );

      field=GetField( Config, From );
      if ( field == NULL )
      {
         FatalError( "Invalid format description line..." );
      }
      new->Stop = atoi( field );

      field=GetField( Config, From );
      if ( field == NULL ||
           strlen( field ) > 1 ||
	   (*field != 'L' && *field != 'N' && *field != 'R'))
      {
	 if ( !field ) FatalError( "Invalid format description line..." );


	 fprintf( stderr, "Type = %s\n", field );
	 FatalError( "Invalid format description line..." );
      }
      new->Type = *field;

      if ( Config->LastColumn )
      {
         Config->LastColumn->Next = new;
      }
      else
      {
         Config->FirstColumn = new;
      }

      Config->LastColumn = new;
   }
}

void MakeSql( CONFIG *Config )
{
   FILE *out;
   char buf[BUFFER+1];
   ColumnDesc *cd;

   if ( strlen( Config->OutputBasename ) > BUFFER - 10 )
   {
      FatalError( "Output basename too long" );
   }

   sprintf( buf, "%s.sql", Config->OutputBasename );

   out = fopen( buf, "w" );
   if ( out == NULL )
   {
      FatalError( "Can't open SQL output file" );
   }

   cd = Config->FirstColumn;

   if ( cd == NULL )
   {
      FatalError ( "Data format description is missing..." );
   }

   fprintf(out,"create table taxdata_%s (\n", Config->TableId );
   fprintf(out,"   taxdatid int8 DEFAULT nextval('taxdatid_seq'::regclass) primary key,\n");
   fprintf(out,"   sourcefilename varchar(100),\n" );

   /* Create table */
   while( cd )
   {

      if ( !strcmp( cd->Name, "StartDateTime") ||
           !strcmp( cd->Name, "EndDateTime"  )  )
      {
         cd->Type = 'T';
      }
      
      switch( cd->Type )
      {
         case  'L':
	    fprintf( out, "   %s varchar(%d)", cd->Name, cd->Stop - cd->Start + 1 );
	    break;

         case  'N':
	    fprintf( out, "   %s varchar(%d)", cd->Name, cd->Stop - cd->Start + 1 );
	    break;

         case  'R':
	    fprintf( out, "   %s int8", cd->Name);
	    break;

         case  'T':
	    fprintf( out, "   %s timestamp", cd->Name);
	    break;

         default:
	    FatalError( "SHOULD NOT OCCUR" );
	    break;
      }

      cd = cd->Next;

      if ( cd ) fprintf ( out, "," );
      fprintf( out, "\n" );
   }

   fprintf( out, ");\n" );

   

   /* Register columns */
   cd = Config->FirstColumn;

   while( cd )
   {
      fprintf( out, "insert into meta_columndesc values ( DEFAULT, %s,\n",
               Config->TableId );
      
      fprintf( out, "   '%s',\n", cd->Name );
      fprintf( out, "   '%c',\n", cd->Type );
      fprintf( out, "   %d );\n", cd->Stop - cd->Start + 1 );

      cd = cd->Next;
   }


   fclose( out );
}


void PrepareDump( CONFIG *Config, FILE *From )
{
   FILE *out;
   char buf[BUFFER+1];
   ColumnDesc *cd;
   int c, i;

   if ( strlen( Config->OutputBasename ) > BUFFER - 10 )
   {
      FatalError( "Output basename too long" );
   }

   sprintf( buf, "%s.data", Config->OutputBasename );

   out = fopen( buf, "w" );
   if ( out == NULL )
   {
      FatalError( "Can't open CSV output file" );
   }

   cd = Config->FirstColumn;
   i=0;

   while( (c = fgetc( From )) != EOF )
   {
      if( c == '\'' ) c = '"';

      if ( i == 0 )
      {

         fprintf( out, "insert into taxdata_%s values (\n", Config->TableId );
	 fprintf( out, "   DEFAULT,\n" );

	 if ( Config->DataFile )
	 {
            fprintf( out, "   '%s',\n", Config->DataFile );
	 }
	 else
	 {
            fprintf( out, "   NULL,\n" );
	 }
      }

      if ( i+1 == cd->Start )
      {
         fprintf ( out, "   " );

	 switch( cd->Type )
	 {
	    case  'L':
	       fprintf( out, "'" );
	       break;

	    case  'N':
	       fprintf( out, "'" );
	       break;

	    case  'R':
	       fprintf( out, "" );
	       break;

	    case  'T':
	       fprintf( out, "to_timestamp('" );
	       break;

	    default:
	       FatalError( "SHOULD NOT OCCUR" );
	       break;
	 }
      }

      fprintf( out, "%c", c );
      ++i;

      if ( i == cd->Stop )
      {
	 switch( cd->Type )
	 {
	    case  'L':
	       fprintf( out, "'" );
	       break;

	    case  'N':
	       fprintf( out, "'" );
	       break;

	    case  'R':
	       fprintf( out, "" );
	       break;

	    case  'T':
	       fprintf( out, "', 'YYYYMMDD HH24:MI:SS')" );
	       break;

	    default:
	       FatalError( "SHOULD NOT OCCUR" );
	       break;
	 }
         cd = cd->Next;

	 if ( cd )
	 {
	    fprintf( out, ",\n" );
	 }
	 else
	 {
	    fprintf ( out, "\n);\n\n" );
	    cd = Config->FirstColumn;
	    i = 0;

	    while( c != EOF && c != '\n' )
	    {
	       c=fgetc( From );
	    }
	 }
      }
   }


   fclose( out );
}


int main( int argc, char **argv )
{
   CONFIG cfg;
   CONFIG *Config;

   Config = &cfg;

   InitConfig( Config, argv[0] );
   ParseOpt( Config, argc, argv );

   /* 
   ** Step 1 : parse format line
   */

   ParseFormat( Config, stdin );

   /*
   ** Step 2 : prepare SQL statements for table creation
   */

   MakeSql( Config );

   /*
   ** Step 3 : save data
   */

   PrepareDump( Config, stdin );

   return 0;
}
