#include <stdio.h>
#include <malloc.h>
#include <unistd.h>
#include <string.h>
#include <stdlib.h>

#include <libpq-fe.h>

#define MAXBUF 10240
#define BIGBUFFER 102400
#define MAX_GEO_SIZE 100

#define CD_FILENAME "SourceFilename"

/*
** Datatypes
*/


typedef enum bool
{
   False=0,
   True=1
}
bool;

typedef struct ColumnDesc
{
   char *Name;
   char *lName;
   int Start;
   int Stop;
   char Type;
   int Storage;

   char *Value;

   //int (*ReworkValue)( CONTEXT *Context, struct ColumnDesc *Self );
   int (*ReworkValue)();

   struct ColumnDesc *Next;
   struct ColumnDesc *Previous;
}
ColumnDesc;

typedef struct basedata
{
   char *Id;
   char *Prefix;
   char *Name;
}
BaseData;

typedef struct OrderedData
{
   BaseData *Infos;

   struct OrderedData *Sons[10];
}
OrderedData;


typedef struct CONTEXT
{
   char *ProgramName;
   char *TaxFilesDir;

   ColumnDesc *FirstColumn;
   ColumnDesc *LastColumn;

   BaseData *GeoData;

   char *DbDesc;

   /* Data needed for Value rewrting process */
   char       *Filename;
   ColumnDesc *CalledNumber;
   ColumnDesc *CallType;
   ColumnDesc *TicketVersion;
   
   PGconn   *Database;
}
CONTEXT;

void DoError( CONTEXT *Context, char *File, int Line, char *Msg )
{
   fprintf( stderr, "%s - %s %d : %s\n",
            (Context->ProgramName==NULL)?"":Context->ProgramName,
	    File,
	    Line,
	    Msg );

}

void DoFatalError( CONTEXT *Context, char *File, int Line, char *Msg )
{
   DoError( Context, File, Line, Msg );
   exit(1);
}

#define FatalError( msg ) DoFatalError( Context, __FILE__, __LINE__, (msg))

#define Error( msg ) DoError( Context, __FILE__, __LINE__, (msg))

#define Debug( msg ) DoError( Context, __FILE__, __LINE__, (msg))

char *DupString( CONTEXT *Context, char *String )
{
   char *new;

   new = strdup( String );

   if ( new == NULL )
   {
      FatalError( "Memory allocation failed" );
   }

   return new;
}

void DisplayShortUsage( CONTEXT *Context, FILE *Out )
{
   fprintf( Out, "TOBEDONE...\n" );
}

void DisplayUsage( CONTEXT *Context, FILE *Out )
{
   fprintf( Out, "TOBEDONE...\n" );
}

void DisplayVersion( CONTEXT *Context, FILE *Out )
{
   fprintf( Out, "TOBEDONE...\n" );
}


void ParseOpt( CONTEXT *Context, int Argc, char **Argv )
{
   int c;
   int err;
   bool help;
   bool vers;
   extern char *optarg;
   extern int optind, opterr, optopt;

   err  = 0;         /* Error count      */
   help = False;     /* Help required    */
   vers = False;     /* Version required */

   while( (c=getopt( Argc, Argv, "hvD:T:" )) > 0 )
   {
      switch( c )
      {
         case 'T':
            Context->TaxFilesDir = DupString( Context, optarg );
	    break;

         case 'D':
            Context->DbDesc = DupString( Context, optarg );
	    break;

	 case 'v' :
	    vers = True;
	    ;;

	 case 'h' :
	    help = True;
	    break;

	 default:
            fprintf( stderr, "Invalid option %c\n", c );
	    ++err;
	    break;
      }
   }

   /*
   ** -D 
   */
   if ( Context->DbDesc == NULL )
   {
      Error( "Database description mandatory" );
      ++err;
   }

   /*
   ** -T 
   */
   if ( Context->TaxFilesDir == NULL )
   {
      Error( "Directory where TAX files are stored is mandatory" );
      ++err;
   }

   /*
   ** Unknown or invalid option encountered
   */
   if ( err )
   {
      DisplayShortUsage( Context, stderr );
      exit( 1 );
   }

   /*
   ** Help required
   */
   if ( help )
   {
      DisplayUsage( Context, stdout );
      exit( 0 );
   }

   /*
   ** Version required
   */
   if ( vers )
   {
      DisplayVersion( Context, stdout );
      exit ( 0 );
   }

}


char *GetField( CONTEXT *Context, char **StrPtr )
{
   static char FieldValue[MAXBUF+1];
   int i, c;
   char *p;

   p = *StrPtr;

   i = 0;

   if( *p == '\0' ) return NULL;

   while ( (c=*p) != '\0' && c != ',' && c != '\n' && c >= '0')
   {
      FieldValue[i++] = c;

      if ( i > MAXBUF )
      {
         FatalError( "Invalid format description line..." );
      }
      ++p;
   }

   if ( c != '\0' ) ++p;

   FieldValue[i] = '\0';

   *StrPtr = p;

   if ( i > 0 ) return FieldValue;

   return NULL;
}

char *ToLower( CONTEXT *Context, char *String )
{
   int i=0;

   if ( !String ) return;

   while( String[i] != '\0' )
   {
      String[i] = tolower( String[i] );
      ++i;
   }

   return String;
}



void DumpDataline( CONTEXT *Context, char *Prefix )
{
   ColumnDesc *cd;
   int i=0;

   cd = Context->FirstColumn;

   while( cd )
   {
      fprintf( stderr, "%s - Col %02d : %s - %s (%d %d %d) = %s (%s <- -> %s )\n",
	 Prefix,
	 i,
         cd->Name,
	 cd->lName,
	 cd->Storage,
	 cd->Start,
	 cd->Stop,
	 (cd->Value==NULL)?"-empty-":cd->Value,
	 (cd->Previous==NULL)?"":cd->Previous->Name,
	 (cd->Next==NULL)?"":cd->Next->Name
	 );

      ++i;
      cd=cd->Next;
   }
}

int ParseFormat( CONTEXT *Context, char *Format )
{
   char *field;
   ColumnDesc *new, *id, *file;
   char *p;

   p = Format;

   ++p;

   while( (field=GetField( Context, &p)) != NULL )
   {
      new = (ColumnDesc *)calloc(1, sizeof(ColumnDesc));

      if( new == NULL )
      {
         Error( "Memory allocation problem..." );
	 return 1;
      }

      new->Name = DupString( Context, field );
      new->lName = ToLower( Context, DupString( Context, new->Name ));


      field=GetField( Context, &p );
      if ( field == NULL )
      {
         Error( "Invalid format description line..." );
	 return 1;
      }
      new->Start = atoi( field );

      field=GetField( Context, &p );
      if ( field == NULL )
      {
         Error( "Invalid format description line..." );
	 return 1;
      }
      new->Stop = atoi( field );

      new->Storage = new->Stop - new->Start + 1;
      new->Value = (char *)calloc(1 + new->Storage, 1 );
      if ( !new->Value )
      {
        FatalError( "Memory :" );
      }


      field=GetField( Context, &p );
      if ( field == NULL ||
           strlen( field ) > 1 ||
	   (*field != 'L' && *field != 'N' && *field != 'R'))
      {
	 if ( !field )
	 {
	    Error( "Invalid format description line..." );
	    return 1;
	 }

	 fprintf( stderr, "Type = %s\n", field );
	 Error( "Invalid format description line..." );
	 return 1;
      }

      if ( !strcmp( new->lName, "startdatetime") ||
           !strcmp( new->lName, "enddatetime"  )  )
      {
         new->Type = 'T';
      }
      else
      {
         new->Type = *field;
      }
      
      new->Next = NULL;

      if ( Context->LastColumn )
      {
         Context->LastColumn->Next = new;
      }
      else
      {
         Context->FirstColumn = new;
      }

      new->Previous = Context->LastColumn;
      Context->LastColumn = new;
   }

//DumpDataline( Context, "PARSEFORMAT" );
   return 0;
}



int RWV_file( CONTEXT *Context, ColumnDesc *Self )
{
   strcpy( Self->Value, Context->Filename );
// fprintf( stderr, "RWV: Copy %s\n", Context->Filename );

   return 0;
}



ColumnDesc *FindColumn( CONTEXT *Context, char *FieldName )
{
   ColumnDesc *cd;
   char buffer[MAXBUF+1];

   strcpy( buffer, FieldName );
   ToLower( Context, buffer );

   cd = Context->FirstColumn;

   while( cd )
   {
      if( !strcmp( cd->lName, buffer ))
      {
         return cd;
      }
      cd = cd->Next;
   }

   return NULL;
}

int ReworkFormat( CONTEXT *Context )
{
   ColumnDesc *file, *id, *cn, *ct, *geo;
   ColumnDesc *p, *n;

   /*
   ** Add pseudocolumns
   */

   file = (ColumnDesc *)calloc(1,sizeof(ColumnDesc));

   if( file == NULL )
   {
      Error( "Memory allocation problem..." );
      return 1;
   }

   file->Name = DupString( Context, CD_FILENAME );
   file->lName = ToLower( Context, DupString( Context,  file->Name ));
   file->Start = file->Stop = 0;
   file->Storage = 100;
   file->Next = Context->FirstColumn;
   file->Next->Previous= file;
   file->Type = 'L';
   file->ReworkValue = RWV_file;

   file->Value = (char *)calloc(1, 1 + file->Storage );
   if ( !file->Value )
   {
     FatalError( "Memory :" );
   }

   id = (ColumnDesc *)calloc(1,sizeof(ColumnDesc));

   if( id == NULL )
   {
      Error( "Memory allocation problem..." );
      return 1;
   }

   id->Name = DupString( Context, "id" );
   id->lName = ToLower( Context, DupString( Context,  id->Name ));
   id->Start = id->Stop = 0;
   id->Storage = 0;
   id->Next = file;
   id->Next->Previous = id;
   id->Previous = NULL;
   id->Type = ' ';
   id->Value = NULL;

   Context->FirstColumn = id;

   geo = (ColumnDesc *)calloc(1,sizeof(ColumnDesc));

   if( geo == NULL )
   {
      Error( "Memory allocation problem..." );
      return 1;
   }

   return 0;
}


void Protected_PQclear( PGresult *Result )
{
   if( Result) PQclear( Result );
}

int UpdateTaxTable( CONTEXT *Context )
{
   char buf[MAXBUF+1];
   char sql[BIGBUFFER+1];
   ColumnDesc *cd;
   PGresult *res, *res2;
   static char tableid[MAXBUF+1];
   char *datatype;
   int size, num;

   cd = Context->FirstColumn;

   if ( cd == NULL )
   {
      Error ( "Data format description is missing..." );
      return 1;
   }

   /* Check and update table */
   while( cd )
   {
      sprintf( sql, "select datatype, size from meta_columndesc where columnname = '%s'", cd->lName );

      res = PQexec( Context->Database, sql );

      if ( res == NULL || PQresultStatus( res ) != PGRES_TUPLES_OK )
      {
         Error ( PQerrorMessage( Context->Database ));
	 Error( "Unable to check column properties" );
	 Protected_PQclear( res );
	 return 1;
      }

      num = PQntuples(res);

      if ( num > 1 )
      {
	 Error( "SHOULD NOT OCCUR" );
	 Protected_PQclear( res );
	 return 1;
      }

      if ( num == 0 )
      {
         /* field does not exist => create and register it */

	 fprintf( stderr, "[New column %s] ", cd->Name );

	 *sql = '\0';

	 sprintf(sql+strlen(sql), "alter table taxdata add column " );

	 switch( cd->Type )
	 {
	    case 'L':
	       sprintf( sql+strlen(sql), "   %s varchar(%d)", cd->lName, cd->Storage );
	       break;

	    case 'N':
	       sprintf( sql+strlen(sql), "   %s varchar(%d)", cd->lName, cd->Storage );
	       break;

	    case 'R':
	    case ' ':
	       sprintf( sql+strlen(sql), "   %s int8", cd->lName);
	       break;

	    case 'T':
	       sprintf( sql+strlen(sql), "   %s timestamp", cd->lName);
	       break;

	    default:
	       FatalError( "SHOULD NOT OCCUR" );
	       break;
	 }

	 res2=PQexec(Context->Database, sql);
      
	 if ( res2 == NULL || PQresultStatus( res2 ) != PGRES_COMMAND_OK )
	 {
            Error ( PQerrorMessage( Context->Database ));
	    Error( "Unable to register new data column" );
	    Protected_PQclear( res2 );
            Protected_PQclear( res );
	    return 1;
	 }
	 Protected_PQclear( res2 );

	 *sql = '\0';
	 sprintf( sql+strlen(sql),
		  "insert into meta_columndesc values ( DEFAULT, '%s',\n",
		  cd->lName );
	 
	 sprintf( sql+strlen(sql), "   '%s',\n", cd->Name );
	 sprintf( sql+strlen(sql), "   '%c',\n", cd->Type );
	 sprintf( sql+strlen(sql), "   %d )", cd->Storage );

	 res2=PQexec(Context->Database, sql);
      
	 if ( res2 == NULL || PQresultStatus( res2 ) != PGRES_COMMAND_OK )
	 {
            Error ( PQerrorMessage( Context->Database ));
	    Error( "Unable to register new data column" );
	    Protected_PQclear( res2 );
            Protected_PQclear( res );
	    return 1;
	 }
	 Protected_PQclear( res2 );
      }
      else
      {
         /* check field type & size */
	 datatype = PQgetvalue( res, 0, 0);
	 size = (PQgetvalue( res, 0, 1)!=NULL)?atoi(PQgetvalue( res, 0, 1)):-1;
	 if ( !datatype || *datatype != cd->Type )
	 {
            if ( *datatype == 'N' || *datatype == 'L' )
	    {
	       cd->Type = *datatype;
	    }
	    else if ( *datatype = 'R' )
	    {
               fprintf ( stderr, "[Change type %s %c=>%c] ", cd->Name, *datatype, cd->Type );
	       sprintf( sql, "alter table taxdata alter %s type varchar(%d)", cd->Name, cd->Storage );
	       res2=PQexec(Context->Database, sql);
	    
	       if ( res2 == NULL || PQresultStatus( res2 ) != PGRES_COMMAND_OK )
	       {
                  Error ( PQerrorMessage( Context->Database ));
		  fprintf( stderr, "Field %s old type = %s - new type = %c\n",
                                   cd->Name, datatype, cd->Type );
		  Error( "Unable to convert column" );
		  Protected_PQclear( res2 );
		  Protected_PQclear( res );
		  return 1;
	       }
	       Protected_PQclear( res2 );

	       sprintf( sql, "update meta_columndesc set (size, datatype) = (%d, '%c' ) where columnname = '%s'", cd->Storage, cd->Type, cd->lName );
	       res2=PQexec(Context->Database, sql);
	    
	       if ( res2 == NULL || PQresultStatus( res2 ) != PGRES_COMMAND_OK )
	       {
                  Error ( PQerrorMessage( Context->Database ));
		  Error( "Unable to register new column size" );
		  Protected_PQclear( res2 );
		  Protected_PQclear( res );
		  return 1;
	       }
	       Protected_PQclear( res2 );
	    }
	    else
	    {
               fprintf( stderr, "Field %s OLD=%s NEW=%c\n", cd->Name, datatype, cd->Type );
	       FatalError( "Column type changed ! Not yet supported" );
	    }
	 }

	 if ( size < cd->Storage )
	 {
            fprintf ( stderr, "[Resize %s %d=>%d] ", cd->Name, size, cd->Storage );
	    sprintf( sql, "alter table taxdata alter %s type varchar(%d)", cd->Name, cd->Storage );
	    res2=PQexec(Context->Database, sql);
	 
	    if ( res2 == NULL || PQresultStatus( res2 ) != PGRES_COMMAND_OK )
	    {
	       Error ( PQerrorMessage( Context->Database ));
	       Error( "Unable to resize column" );
	       Protected_PQclear( res2 );
               Protected_PQclear( res );
	       return 1;
	    }
	    Protected_PQclear( res2 );

	    sprintf( sql, "update meta_columndesc set size = %d where columnname = '%s'", cd->Storage, cd->lName );
	    res2=PQexec(Context->Database, sql);
	 
	    if ( res2 == NULL || PQresultStatus( res2 ) != PGRES_COMMAND_OK )
	    {
	       Error ( PQerrorMessage( Context->Database ));
	       Error( "Unable to register new column size" );
	       Protected_PQclear( res2 );
               Protected_PQclear( res );
	       return 1;
	    }
	    Protected_PQclear( res2 );
	 }
      }

      Protected_PQclear( res );

      cd = cd->Next;
   }

   return 0;
}




void InitContext( CONTEXT *Context, char *ProgramName )
{
   memset( Context, 0, sizeof( CONTEXT ));

   Context->ProgramName = DupString(Context, ProgramName);

}

char *DbStatus( CONTEXT *Context )
{
   if ( Context->Database == NULL )
   {
      return "No database descriptor";
   }

   switch ( PQstatus(Context->Database))
   {
      case CONNECTION_OK:
         return "Connexion ok.";

      case CONNECTION_BAD:
         return "Connexion failed.";

      case CONNECTION_STARTED:
         return "Waiting for connection to be made.";

      case CONNECTION_MADE:
         return "Connection OK; waiting to send.";

      case CONNECTION_AWAITING_RESPONSE:
         return "Waiting for a response from the server.";

      case CONNECTION_AUTH_OK:
         return "Received authentication; waiting for backend start-up to finish.";

      case CONNECTION_SSL_STARTUP:
         return "Negotiating SSL encryption.";

      case CONNECTION_SETENV:
         return "Negotiating environment-driven parameter settings.";
   }

   return "Unknown status - SHOULD NOT OCCUR !";

}

void OpenDb( CONTEXT *Context )
{
   char buffer[MAXBUF+1];

   Context->Database = PQconnectdb( Context->DbDesc );

   if ( Context->Database == NULL || PQstatus(Context->Database) == CONNECTION_BAD)
   {
      if ( Context->Database ) Error ( PQerrorMessage( Context->Database ));

      FatalError( "Can't connect to postgres database !" );
   }
}


char *ReadLine( CONTEXT *Context, FILE *From )
{
   static char buffer[BIGBUFFER+1];
   int i;
   char *res;

   res = fgets( buffer, BIGBUFFER, From );

   if( res == NULL )
   {
      return NULL;
   }

   if( strlen(buffer) >= BIGBUFFER )
   {
      Error( "Line too long" );
      return NULL;
   }

   if ( !*buffer ) 
   {
      return NULL;
   }

   i = 0;
   while( buffer[i] )
   {
      if( buffer[i]=='\n' )
      {
         buffer[i]='\0';
	 break;
      }
      ++i;
   }

   return buffer;
}

void ComputeFilelist( CONTEXT *Context )
{
   PGresult *res, *res2;
   char buffer[MAXBUF+1];
   FILE *flist;
   char *file;
   int i;
   int num;

   /* Begin transaction */

   res = PQexec( Context->Database, "BEGIN" );

   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res );
      FatalError( "Unable to begin transaction for building filelist" );
   }

   Protected_PQclear( res );

   /* truncate table filelist */

   res = PQexec( Context->Database, "truncate table spooledfiles;" );

   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res );
      FatalError( "Unable to truncate filelist table" );
   }

   Protected_PQclear( res );

   /* start find on TAX dir */

   fprintf( stderr, "Listing files to import... " );
   sprintf( buffer, "find %s -maxdepth 1 -type f -name \"TAX*.DAT\" -printf \"%%f\n\"",
                    Context->TaxFilesDir );
   flist = popen(buffer, "r");

   if ( flist == NULL )
   {
      FatalError ( "Unable to get taxfile list" );
   }

   /* insert filenames into table */
   while( (file=ReadLine( Context, flist )) != NULL )
   {
      sprintf( buffer, "insert into spooledfiles values ( '%s' )", file );
      res = PQexec( Context->Database, buffer );

      if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
      {
         Error ( PQerrorMessage( Context->Database ));
         Protected_PQclear( res );
         FatalError( "Unable to insert filename into filelist" );
      }
      Protected_PQclear( res );
   }

   pclose( flist);

   /* remove already imported filenames from list */

   sprintf( buffer, "delete from spooledfiles where filename in (select distinct %s from taxdata)", CD_FILENAME);

   res = PQexec( Context->Database, buffer );
   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res );
      FatalError( "Unable to clean filelist" );
   }
   Protected_PQclear( res );

   /* commit transaction */

   res = PQexec( Context->Database, "COMMIT" );

   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res );
      FatalError( "Unable to commit filelist transaction" );
   }

   Protected_PQclear( res );
}

void CleanupColumnDesc( CONTEXT *Context )
{
   ColumnDesc *cd, *next;

   cd = Context->FirstColumn;

   while( cd )
   {
      next = cd->Next;

      if( cd->lName ) free( cd->lName );
      cd->lName = NULL;

      if( cd->Name ) free( cd->Name );
      cd->Name = NULL;

      if( cd->Value ) free( cd->Value );
      cd->Value = NULL;

      cd = next;
   }

   Context->FirstColumn = NULL;
   Context->LastColumn = NULL;
}

int ParseData( CONTEXT *Context, char *Line )
{
   int i, j;
   ColumnDesc *cd;
   char *p;

   p = Line;
   i = 1;
   cd = Context->FirstColumn;

   j=0;
   while( *p )
   {
      while( cd && cd->Stop < i )
      {
         cd = cd->Next; j=0;
      }

      if( !cd ) 
      {
         Error( "Too many data in row" );
	 return 1;
      }

      cd->Value[j++] = *p;
      cd->Value[j] = '\0';

      ++i;
      ++p;
   }

   if( cd->Next ) 
   {
      Error( "Too few data in row" );
      return 1;
   }

   return 0;
//DumpDataline( Context, "PARSEDATA" );
}

void RemoveUselessSpaces( CONTEXT *Context, char *String )
{
   char *p, *q;

   if( !String ) return;

   p = String;

   /* remove leading spaces */
   while ( *p == ' ' )
   {
      ++p;
   }

   if ( p != String )
   {
      q = String;

      while ( *p )
      {
         *q++ = *p++;
      }
      *q = '\0';
   }

   if ( strlen( String ) )
   {
      /* remove trailing spaces */

      p = String + strlen( String ) - 1;

      while ( p >= String && *p == ' ' )
      {
         --p;
      }
      ++p;
      *p = '\0';
   }
}

int ReworkData( CONTEXT *Context, char *Filename ) 
{
   ColumnDesc *cd;
//DumpDataline( Context, "REWORKDATA DEBUT" );

   Context->Filename = Filename;

   cd = Context->FirstColumn;

   while( cd )
   {
      if ( cd->ReworkValue )
      {
// fprintf( stderr, "Calling rework method for %s\n", cd->Name );
         if ( (*cd->ReworkValue)( Context, cd ) )
	 {
	    Error( "Rework function failed" );
	    return 1;
	 }
      }
//fprintf( stderr, "Removing spaces for %s '%s'\n", cd->Name, cd->Value );
      RemoveUselessSpaces( Context, cd->Value);
//fprintf( stderr, "Rework done for %s '%s'\n", cd->Name , cd->Value);

      cd = cd->Next;
   }
//DumpDataline( Context, "REWORKDATA FIN" );

   return 0;
}

void AddValue( CONTEXT *Context, char *Sql, char *Value )
{
   char *p, *q;

   p = Sql+strlen(Sql);

   q = Value;

   while( *q )
   {
      if ( *q == '\'' ) *p++ = *q;
      *p++ = *q++;
   }

   *p = '\0';
}

int InsertDataIntoDatabase( CONTEXT *Context )
{
   char sql[BIGBUFFER + 1];
   char zone[BIGBUFFER + 1];
   ColumnDesc *cd;
   PGresult *res = NULL, *res1 = NULL;
   int num;
   char *id;
   char *number;
   ColumnDesc *cn, *ct, *tv;
   char *nbr;
   int i,l;
   char *desc;

   cd = Context->FirstColumn;

   if( !cd ) return 1;

   /*
   ** Get entry id
   */

   res1 =  PQexec( Context->Database, "select nextval('taxdata_id_seq'::regclass)" );
   if ( res1 == NULL || PQresultStatus( res1 ) != PGRES_TUPLES_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res1 );
      Error( "Unable to insert data" );
      return 1;
   }

   num = PQntuples(res1);

   if ( num != 1 )
   {
      Protected_PQclear( res1 );
      Error( "Unable to get new ticket Id" );
      return 1;
   }

   id = PQgetvalue( res1, 0, 0);

   *sql = '\0';

   sprintf( sql, "insert into taxdata (" );

   while ( cd )
   {
      sprintf(sql+strlen(sql), "%s", cd->Name );

      if( cd->Next ) sprintf( sql+strlen(sql), ", " );

      cd = cd->Next;
   }

   sprintf( sql+strlen(sql), ") values (" );

   cd = Context->FirstColumn;
   while ( cd )
   {
      switch(cd->Type )
      {
         case ' ':
	    sprintf(sql+strlen(sql), "%s", id );
	    break;

         case 'L':
         case 'N':
	    sprintf(sql+strlen(sql), "'" );
	    AddValue( Context, sql, cd->Value );
	    sprintf(sql+strlen(sql), "'" );
	    break;

         case 'R':
	    AddValue( Context, sql, cd->Value );
	    break;

         case 'T':
	    sprintf(sql+strlen(sql), "to_timestamp( '%s', 'YYYYMMDD HH24:MI:SS')", cd->Value );
	    break;

	 default:
	    FatalError( "SHOULD NOT OCCUR" );
	    break;
      }

      if( cd->Next ) sprintf( sql+strlen(sql), ", " );

      cd = cd->Next;
   }

   sprintf( sql+strlen(sql), ")" );

   res = PQexec( Context->Database, sql );
   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
Error( sql );
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res1 );
      Protected_PQclear( res );
      Error( "Unable to insert data" );
exit(1);
      return 1;
   }
   Protected_PQclear( res );

   cn = FindColumn( Context, "CalledNumber" );
   ct = FindColumn( Context, "CallType" );
   tv = FindColumn( Context, "TicketVersion" );

   if ( cn == NULL || ct == NULL || tv == NULL )
   {
      FatalError( "SHOULD NOT OCCUR");
   }
   if ( cn->Value == NULL || ct->Value == NULL || tv->Value == NULL )
   {
      FatalError( "SHOULD NOT OCCUR");
   }

   nbr = cn->Value;
   if ( strcmp( tv->Value, "ED4.8" ) <= 0 )
   {
      if ( *ct->Value == '4' )
      {
         ++nbr;
      }
   }

   l = strlen( nbr );

   sprintf( sql, "select %s, substring( '%s', 1, %d - length(regexp_replace( '%s', regexp_prefix, '' ))), %d - length(regexp_replace( '%s', regexp_prefix, '' )), level, explanation from geozones_list where '%s' ~ regexp_prefix order by level",
      id, nbr, l, nbr, l, nbr, nbr );

   res = PQexec( Context->Database, sql );
   
   if ( res == NULL || PQresultStatus( res ) != PGRES_TUPLES_OK )
   {
Error( sql );
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res1 );
      Protected_PQclear( res );
      Error( "Unable to insert data" );
      return 1;
   }

   num = PQntuples(res);

   if ( num != 0 )
   {
      Protected_PQclear( res1 );

      *zone = '\0';

      for(i=0;i<num;++i)
      {
	 AddValue( Context, zone, "[" );
         AddValue( Context, zone, PQgetvalue( res, i, 4) );
	 AddValue( Context, zone, "]" );

	 sprintf( sql, "insert into geozones values ( DEFAULT, %s, '%s', %s, %s,%d, '%s', '",
	          PQgetvalue( res, i, 0),   // taxdataid
	          PQgetvalue( res, i, 1),   // match
	          PQgetvalue( res, i, 2),   // lmatch
	          PQgetvalue( res, i, 3),   // level
                  (i==(num-1)),		    // niveau final
		  zone );

	 AddValue( Context, sql, PQgetvalue( res, i, 4) );

         sprintf( sql+strlen(sql), "' )" );

	 res1 = PQexec( Context->Database, sql );

         if ( res1 == NULL || PQresultStatus( res1 ) != PGRES_COMMAND_OK )
         {
            Error ( PQerrorMessage( Context->Database ));
            Protected_PQclear( res1 );
            Protected_PQclear( res );
            Error( "Unable to insert data" );
            return 1;
         }

	 Protected_PQclear( res1 );
      }

      Protected_PQclear( res );
      return 0;
   }

   if( *nbr < '0' || *nbr > '9' )
   {
      desc="NONE";
   }
   else
   {
      desc="UNKNOWN";
   }

   sprintf( sql, "insert into geozones values ( DEFAULT, %s, '', 0, 0, 1, '[%s]', '%s')", id, desc, desc );

   res = PQexec( Context->Database, sql );
   
   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
Error( sql );
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res1 );
      Protected_PQclear( res );
      Error( "Unable to insert data" );
exit(1);
      return 1;
   }

   Protected_PQclear( res );
   Protected_PQclear( res1 );

   return 0;
}


int ImportFile( CONTEXT *Context, char *Filename )
{
   PGresult *res, *res2;
   char buffer[MAXBUF+1];
   FILE *tax;
   char *line;
   int i;
   int num;
   int err = 0;
   char *sign = NULL;
   int status;

   /* Begin transaction */

   res = PQexec( Context->Database, "BEGIN" );

   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res );
      FatalError( "Unable to begin transaction for importing file" );
   }

   Protected_PQclear( res );

   sprintf( buffer, "zcat %s/%s", Context->TaxFilesDir, Filename );
   tax = popen(buffer, "r");

   if ( tax == NULL )
   {
      Error( buffer );
      Error( "Can't open file for reading..." );
      ++err;
   }
   else
   {
      line = ReadLine( Context, tax);

      if ( line == NULL || *line != '#' )
      {
	 Error( buffer );
	 Error( "Not format description in file..." );
         ++err;
      }
      else
      {
         status = ParseFormat( Context, line );

         if( status == 0 )
	 {
	    status = ReworkFormat( Context );

	    if ( status == 0 )
	    {
	       status = UpdateTaxTable( Context );
	       if ( status != 0 )
	       {
		  ++err;
	       }
	       else
	       {
		  int nok, nko;

		  nok = nko = 0;

		  while( (line = ReadLine( Context, tax)) != NULL && *line != '\0' )
		  {
		     if( ParseData( Context, line ) )
		     {
			++err;
			++nko;
		     }
		     else if ( ReworkData( Context, Filename ) )
		     {
			++err;
			++nko;
		     }
		     else if ( InsertDataIntoDatabase( Context ) )
		     {
			++err;
			++nko;
		     }
		     else
		     {
			++nok;
		     }
		  }
		  fprintf( stderr, "%d line%s (%d NOK) ",
			   nok+nko, ((nok+nko)>1)?"s":"", nko);
	       }
	    }
	    else
	    {
	       Error( "Can't rework format..." );
	       ++err;
	    }
	 }
	 else
	 {
	    Error( "Format parsing failed !" );
	    ++err;
	 }
	 /* FREE DATA STRUCTURE */
	 CleanupColumnDesc( Context );
      }

      pclose( tax );
   }


   /* commit or cancel transaction */

   if ( err == 0 )
   {
      res = PQexec( Context->Database, "COMMIT" );
   }
   else
   {
      res = PQexec( Context->Database, "ROLLBACK" );
   }

   if ( res == NULL || PQresultStatus( res ) != PGRES_COMMAND_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res );
      FatalError( "Unable to commit file import transaction" );
   }

   Protected_PQclear( res );

   return (err==0)?0:1;
}

int ImportFiles( CONTEXT *Context )
{
   PGresult *res, *res2;
   char buffer[MAXBUF+1];
   FILE *flist;
   char *file;
   int i;
   int num;
   int s, err = 0;

   res = PQexec( Context->Database, "select filename from spooledfiles order by filename" );
   if ( res == NULL || PQresultStatus( res ) != PGRES_TUPLES_OK )
   {
      Error ( PQerrorMessage( Context->Database ));
      Protected_PQclear( res );
      FatalError( "Unable to get file list" );
   }

   num = PQntuples(res);
   fprintf( stderr, "%d files to import...\n", num );

   for( i = 0; i < num; ++i)
   {
      fprintf( stderr, "%d/%d : Importing %s... ", i+1, num, PQgetvalue( res, i, 0));
      s = ImportFile( Context, PQgetvalue( res, i, 0) );
      err += s;
      if ( s )
      {
         fprintf( stderr, "FAILED !\n" );
      }
      else
      {
         fprintf( stderr, "Done\n" );
      }
   }

   Protected_PQclear( res );

   return err;
}

int main( int argc, char **argv )
{
   CONTEXT ctx;
   CONTEXT *Context;
   int i;
   FILE *tax;
   char buffer[MAXBUF+1];
   char *line;
   int status;

   Context = &ctx;

   /*
   ** Initialize storage areas
   */

   InitContext( Context, argv[0] );

   /*
   ** Parse command line options
   */

   ParseOpt( Context, argc, argv );

   /*
   ** Connect to Postgres database
   */

   OpenDb( Context );

   /*
   ** List files in dir, insert list in DB and compute which to import
   */

   ComputeFilelist( Context );

   /*
   ** Import listed files
   */

   status = ImportFiles( Context );

   /*
   ** Vacuum tables
   */

   /* VacuumDataTables( Context ); */

   exit(status);
}
